﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;

namespace CanterburySharp
{
	public class Conversation : NotifyableObject
	{
		private List<Edge> mEdges = new List<Edge>();
		private List<Line> mVertices = new List<Line>();
		private List<Direction> mStageDirections = new List<Direction>();

		private string mName = "";

		public List<Edge> Replies { get => mEdges; set { mEdges = value; RaisePropertyChange("Edges"); } }
		public List<Line> Lines { get => mVertices; set { mVertices = value; RaisePropertyChange("Vertices"); } }
		public List<Direction> StageDirections { get => mStageDirections; set { mStageDirections = value; RaisePropertyChange("StageDirections"); } }
		public string Name { get => mName; set { mName = value; RaisePropertyChange("Name"); } }
		
		/// <summary>
		/// Copy construct
		/// </summary>
		/// <param name="convo"></param>
		public Conversation(Conversation convo)
		{
			Name = convo.Name;
			foreach (var line in convo.Lines)
			{
				Lines.Add(new Line(line));
			}

			foreach (var edge in convo.Replies)
			{
				Replies.Add(new Edge(edge));
			}


			foreach (var stage in convo.StageDirections)
			{
				StageDirections.Add(new Direction(stage));
			}

			int i = 0;
			foreach (var reply in Replies)
			{
				var src = Lines.Find(x => x.Guid.Equals(reply.SourceGuid));
				if (src != null)
				{
					reply.Source = src;
					src.AddOutputEdge(reply);
				}

				var dst = Lines.Find(x => x.Guid.Equals(reply.DestinationGuid));
				if (dst != null)
				{
					reply.Destination = dst;
					dst.AddInputEdge(reply);
				}
				i++;
			}
		}

		/// <summary>
		/// Equality operator
		/// </summary>
		/// <param name="convo"></param>
		/// <returns></returns>
		public bool AreSame(Conversation convo)
		{
			if (convo.Lines.Count != Lines.Count || convo.Replies.Count != Replies.Count)
			{
				return false;
			}

			for (int i = 0; i < convo.Lines.Count; i++)
			{
				if (convo.Lines[i].Inputs.Count != Lines[i].Inputs.Count ||
					convo.Lines[i].Outputs.Count != Lines[i].Outputs.Count ||
					convo.Lines[i].OutputDirections.Count != Lines[i].OutputDirections.Count ||
					convo.Lines[i].InputDirections.Count != Lines[i].InputDirections.Count ||
					convo.Lines[i].Root != Lines[i].Root ||
					!convo.Lines[i].Dialogue.Equals(Lines[i].Dialogue) ||
					!convo.Lines[i].Speaker.Equals(Lines[i].Speaker))
				{
					return false;
				}

				for (int j = 0; j < convo.Lines[i].Inputs.Count; j++)
				{
					if (convo.Lines[i].Inputs[j].SourceGuid != Lines[i].Inputs[j].SourceGuid ||
						convo.Lines[i].Inputs[j].DestinationGuid != Lines[i].Inputs[j].DestinationGuid ||
						convo.Lines[i].Inputs[j].Guid != Lines[i].Inputs[j].Guid)
					{
						return false;
					}
				}
				for (int j = 0; j < convo.Lines[i].Outputs.Count; j++)
				{
					if (convo.Lines[i].Outputs[j].SourceGuid != Lines[i].Outputs[j].SourceGuid ||
						convo.Lines[i].Outputs[j].DestinationGuid != Lines[i].Outputs[j].DestinationGuid ||
						convo.Lines[i].Outputs[j].Guid != Lines[i].Outputs[j].Guid)
					{
						return false;
					}
				}
				for (int j = 0; j < convo.Lines[i].OutputDirections.Count; j++)
				{
					if (convo.Lines[i].OutputDirections[j].DirectionName != Lines[i].OutputDirections[j].DirectionName ||
						convo.Lines[i].OutputDirections[j].DirectionValue != Lines[i].OutputDirections[j].DirectionValue ||
						convo.Lines[i].OutputDirections[j].AssignmentValue != Lines[i].OutputDirections[j].AssignmentValue)
					{
						return false;
					}
				}
				for (int j = 0; j < convo.Lines[i].InputDirections.Count; j++)
				{
					if (convo.Lines[i].InputDirections[j].DirectionName != Lines[i].InputDirections[j].DirectionName ||
						convo.Lines[i].InputDirections[j].DirectionValue != Lines[i].InputDirections[j].DirectionValue ||
						convo.Lines[i].InputDirections[j].LogicValue != Lines[i].InputDirections[j].LogicValue ||
						convo.Lines[i].InputDirections[j].ComparisonValue != Lines[i].InputDirections[j].ComparisonValue)
					{
						return false;
					}
				}
			}

			return true;
		}

		[OnDeserialized]
		internal void OnDeserializedMethod(StreamingContext context)
		{
			if (Lines.Count == 0)
			{
				InventLine();

				Lines[0].Speaker = "Josh";
				Lines[0].Dialogue = "Welcome to the prototype text interface! Start screwing around to see what happens.";

			}

			List<string> directions = new List<string>();
			foreach (var str in StageDirections)
			{
				directions.Add(str.DirectionName);
			}


			foreach (var line in Lines)
			{
				foreach (var input in line.InputDirections)
					input.DirectionStrings = directions;
				foreach (var output in line.OutputDirections)
					output.DirectionStrings = directions;
			}

			// TO DO - cut the index from parsing, find in list and pick index that way 

				int i = 0;
			foreach (var reply in Replies)
			{
				var src = Lines.Find(x => x.Guid.Equals(reply.SourceGuid));
				if (src != null)
				{
					reply.Source = src;
					src.AddOutputEdge(reply);
				}

				var dst = Lines.Find(x => x.Guid.Equals(reply.DestinationGuid));
				if (dst != null)
				{
					reply.Destination = dst;
					dst.AddInputEdge(reply);
				}
				i++;
			}
		}

		public Conversation(string name)
		{
			mName = name;
		}

		/// <summary>
		/// Make a new edge 
		/// </summary>
		/// <returns></returns>
		public Edge InventEdge()
		{
			Edge reply = new Edge();

			mEdges.Add(reply);
			RaisePropertyChange("Edges");

			return reply;
		}

		/// <summary>
		/// Make a new line
		/// </summary>
		/// <returns></returns>
		public Line InventLine()
		{
			Line line = new Line();

			mVertices.Add(line);
			RaisePropertyChange("Vertices");

			return line;
		}

		/// <summary>
		/// Set up input/output for the edge
		/// </summary>
		/// <param name="source"></param>
		/// <param name="destination"></param>
		public void MakeEdge(Line source, Line destination)
		{
			if (mEdges.Find(x => x.Source == source && x.Destination == destination) != null)
				return;

			Edge edge = new Edge();

			edge.Destination = destination;
			destination.AddInputEdge(edge);
			edge.Source = source;
			source.AddOutputEdge(edge);

			mEdges.Add(edge);

			RaisePropertyChange("Vertices");
			RaisePropertyChange("Edges");

		}

		/// <summary>
		/// Nuke this edge
		/// </summary>
		/// <param name="edge"></param>
		public void DestroyEdge(Edge edge)
		{
			if (edge.Source != null)
				edge.Source.RemoveOutputEdge(edge);
			if (edge.Destination != null)
				edge.Destination.RemoveInputEdge(edge);
			mEdges.Remove(edge);

			RaisePropertyChange("Edges");
			RaisePropertyChange("Vertices");
		}

		/// <summary>
		/// Get the index of a given line 
		/// </summary>
		/// <param name="vertex"></param>
		/// <returns></returns>
		public int GetIndexLines(Vertex vertex)
		{
			return mVertices.IndexOf((Line)vertex);
		}

		/// <summary>
		/// Get the index of a given line's guid
		/// </summary>
		/// <param name="guid"></param>
		/// <returns></returns>
		public int GetIndexLines(Guid guid)
		{
			int ret = -1;
			for (int i = 0; i < mVertices.Count; ++i)
			{
				if (mVertices[i].Guid == guid)
					ret = i;
			}

			return ret;
		}

		/// <summary>
		/// Add a new direction
		/// </summary>
		/// <param name="direction"></param>
		public void AddDirection(Direction direction)
		{
			mStageDirections.Add(direction);
			RaisePropertyChange("StageDirections");

		}

		/// <summary>
		/// Delete this direction
		/// </summary>
		/// <param name="index"></param>
		public void DeleteDirection(int index)
		{
			mStageDirections.RemoveAt(index);
			RaisePropertyChange("StageDirections");
		}

		/// <summary>
		/// Destroy this line 
		/// </summary>
		/// <param name="vertex"></param>
		public void DestroyLine(Line vertex)
		{
			while (vertex.Inputs.Count > 0)
			{
				DestroyEdge((Edge)vertex.Inputs[0]);
				//input.Destination = null;
			}

			while (vertex.Outputs.Count > 0)
			{
				DestroyEdge((Edge)vertex.Outputs[0]);
				//output.Source = null;
			}

			mVertices.Remove(vertex);

			RaisePropertyChange("Edges");
			RaisePropertyChange("Vertices");
		}
	}
}
