﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace CanterburySharp
{
	/// <summary>
	/// Interaction logic for DisplayPage.xaml
	/// </summary>
	public partial class DisplayPage : Page
	{
		List<DisplayLine> mDisplayLines = new List<DisplayLine>();
		List<DisplayEdge> mDisplayReplies = new List<DisplayEdge>();

		MainWindow mRootWindow;

		bool mAddingMode = false;

		bool mMultipleSelection = false;
		bool mControlMode = false;

		Guid mTempEdgeSource;

		Point mStartPos;

		float scale = 1.0f;
		float zoomInMax = 2.0f;
		float zoomOutMax = 0.5f;

		DisplayLine mCopiedDisplayLine = null;

		/// <summary>
		/// Set the parentage index for all the elements
		/// </summary>
		/// <param name="convo"></param>
		/// <returns></returns>
		private Conversation SetParents(Conversation convo)
		{
			foreach (Line line in convo.Lines)
			{
				line.ParentChain = 0;
			}

			List<Guid> found = new List<Guid>();

			foreach (var root in convo.Lines.FindAll(x => x.Root))
				root.FindParentagePosition(found);
			foreach (var root in convo.Lines.FindAll(x => x.Inputs.Count == 0))
				root.FindParentagePosition(found);

			return convo;
		}

		/// <summary>
		/// Add line to the display list
		/// </summary>
		/// <param name="line"></param>
		/// <returns></returns>
		private DisplayLine MakeDisplayLine(Line line)
		{
			var tempLine = new DisplayLine(line);
			tempLine.MouseDown += DisplayLineGrabbed;
			tempLine.MouseUp += DisplayLineReleased;
			tempLine.DialogueText.TextChanged += LineTextChanged;
			tempLine.SpeakerText.TextChanged += LineTextChanged;
			tempLine.Delete.Click += DeleteLineInMain;
			tempLine.Connect.Click += AddNewEdge;
			tempLine.LoadIntoMain.Click += LoadOnMain;
			tempLine.Copy.Click += CopyLine;

			tempLine.DataContext = line;

			return tempLine;
		}

		/// <summary>
		/// Add lines to the display list
		/// </summary>
		/// <param name="convo"></param>
		/// <param name="index"></param>
		private void MakeDisplayLines(Conversation convo, int index)
		{
			int counter = 0;
			foreach (Line line in convo.Lines)
			{
				var dpl = MakeDisplayLine(line);
				mDisplayLines.Add(dpl);
				if (index == counter)
				{
					dpl.LoadUp(true);
				}
				counter++;

			}
		}

		/// <summary>
		/// Add reply to the display list
		/// </summary>
		/// <param name="reply"></param>
		/// <returns></returns>
		private DisplayEdge MakeDisplayReply(Edge reply)
		{
			var tempEdge = new DisplayEdge(reply);
			if (reply.Source != null && reply.Destination != null)
			{
				var source = mDisplayLines.Find(x => x.Guid == ((Line)reply.Source).Guid);
				if (source != null)
				{
					source.AddAsSource(tempEdge);
					tempEdge.DisplaySource = source;
				}
				var destiny = mDisplayLines.Find(x => x.Guid == ((Line)reply.Destination).Guid);
				if (destiny != null)
				{
					destiny.AddAsDestination(tempEdge);
					tempEdge.DisplayDestination = destiny;
				}
			}
			if (tempEdge.DisplaySource != null && tempEdge.DisplayDestination != null)
			{
				tempEdge.Delete.Click += DeleteEdgeInMain;
			}

			return tempEdge;
		}

		/// <summary>
		/// Add replies to the display list
		/// </summary>
		/// <param name="convo"></param>
		private void MakeDisplayReplies(Conversation convo)
		{
			foreach (Edge reply in convo.Replies)
			{
				mDisplayReplies.Add(MakeDisplayReply(reply));
			}
		}

		/// <summary>
		/// Find which layer has the most elements
		/// </summary>
		/// <param name="convo"></param>
		/// <returns></returns>
		private int FindBiggerLayer(Conversation convo)
		{
			int maxSize = 0;
			for (int i = convo.Lines.Count - 1; i >= 0; --i)
			{
				maxSize = Math.Max(maxSize, convo.Lines.FindAll(x => x.ParentChain == i).Count);
			}
			return maxSize;
		}

		/// <summary>
		/// Move the line positioning
		/// </summary>
		/// <param name="convo"></param>
		/// <param name="maxSize"></param>
		private void PositionLines(Conversation convo, int maxSize)
		{
			int verticalMultiplier = 250;
			int horizontalMultiplier = 300;

			//	For each line level - todo: don't do this because it's not good
			for (int i = convo.Lines.Count + 2; i >= 0; --i)
			{
				var founds = convo.Lines.FindAll(x => x.ParentChain == i);
				if (founds.Count == 0)
				{
					continue;
				}

				int j = 0;
				int totalAtlevel = founds.Count;
				foreach (var found in founds)
				{
					var curDisplayLine = mDisplayLines.Find(x => x.Guid == found.Guid);
					MyCanvas.Children.Add(curDisplayLine);

					Canvas.SetTop(curDisplayLine, (i) * verticalMultiplier);

					double offset = (0.5 * (maxSize - totalAtlevel)) + 0.5;

					Canvas.SetLeft(curDisplayLine, ((j + offset) * horizontalMultiplier));

					j++;
				}
			}
		}

		/// <summary>
		/// Move the reply positioning
		/// </summary>
		/// <param name="convo"></param>
		private void PositionReplies(Conversation convo)
		{
			foreach (var line in mDisplayLines)
				line.SetEdgesInit();

			foreach (var reply in mDisplayReplies)
			{
				if (reply.DisplaySource != null && reply.DisplayDestination != null)
				{
					MyCanvas.Children.Add(reply);
				}
			}
		}

		public DisplayPage(MainWindow main)
		{
			InitializeComponent();
			PreviewMouseMove += DisplayLineMoved;
			ReloadDisplay(main);
		}

		/// <summary>
		/// Mouse moved while we are grabbing something
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DisplayLineMoved(object sender, MouseEventArgs e)
		{
			if (e.LeftButton == MouseButtonState.Pressed)
			{
				// Get the current mouse position
				Point curPos = e.GetPosition(null);

				Point newPos = new Point((curPos.X / scale) - (mStartPos.X / scale), (curPos.Y / scale) - (mStartPos.Y / scale));
				mStartPos = curPos;

				foreach (DisplayLine dpl in mDisplayLines.FindAll(x => x.SelectedByMouse))
				{
					var vo = MainScroll.VerticalOffset;
					var ho = MainScroll.HorizontalOffset;

					double left = Canvas.GetLeft(dpl);
					double top = Canvas.GetTop(dpl);

					double newX = (left) + newPos.X;
					double newY = (top) + (newPos.Y);

					Canvas.SetLeft(dpl, Math.Max(0, newX));
					Canvas.SetTop(dpl, Math.Max(0, newY));

					dpl.UpdateInputs();
					dpl.UpdateOutputs();


				}
			}
		}

		/// <summary>
		/// Line has been grabbed
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DisplayLineGrabbed(object sender, MouseButtonEventArgs e)
		{
			DisplayLine dpl = ((DisplayLine)(e.Source));
			dpl.Selection(!dpl.SelectedByMouse);
			mStartPos = e.GetPosition(null);
		}

		/// <summary>
		/// Line has been released
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DisplayLineReleased(object sender, MouseButtonEventArgs e)
		{
			DisplayLine dpl = ((DisplayLine)(e.Source));
			if (!mMultipleSelection)
				dpl.Selection(false);
		}

		/// <summary>
		/// Text changed in a line
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void LineTextChanged(object sender, TextChangedEventArgs e)
		{
			mRootWindow.UpdateUndoList();
			mRootWindow.LoadInComboBoxes();
		}

		/// <summary>
		/// Mimic line deletion in focus
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DeleteLineInMain(object sender, RoutedEventArgs e)
		{
			mRootWindow.UpdateUndoList();
			Button b = (Button)(sender);
			DisplayLine dpl = (DisplayLine)VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(b))));

			if (mRootWindow.GetConversation().Lines.Count == 1)
				return;

			foreach (var rep in mDisplayReplies)
			{
				if (rep.DisplaySource == dpl || rep.DisplayDestination == dpl)
				{
					MyCanvas.Children.Remove(rep);
				}
			}
			MyCanvas.Children.Remove(dpl);
			mRootWindow.GetConversation().DestroyLine(mRootWindow.GetConversation().Lines.Find(x => x.Guid == dpl.Guid));

			UpdateMainConvo();

			DisplayLine temp = mDisplayLines.Find(x => x.Guid == mRootWindow.GetConversation().Lines[mRootWindow.focusPage.CurrentIndex].Guid);
			temp.LoadUp(true);
		}

		/// <summary>
		/// Mimic edge deletion in focus
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DeleteEdgeInMain(object sender, RoutedEventArgs e)
		{
			mRootWindow.UpdateUndoList();
			Button b = (Button)(sender);
			DisplayEdge dpr = (DisplayEdge)VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(b))));
			MyCanvas.Children.Remove(dpr);

			var src = dpr.DisplaySource;
			var dst = dpr.DisplayDestination;

			src.RemoveAsSource(dpr);
			dst.RemoveAsDestination(dpr);

			Edge rep = mRootWindow.GetConversation().Replies.Find(x => x.Guid == dpr.Guid);

			mRootWindow.GetConversation().DestroyEdge(rep);

			UpdateMainConvo();
		}

		/// <summary>
		/// Load up the conversation in the focus
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void LoadOnMain(object sender, RoutedEventArgs e)
		{
			Button b = (Button)(sender);
			DisplayLine dpl = (DisplayLine)VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(b))));

			foreach (var d in mDisplayLines)
			{
				d.LoadUp(false);
			}

			dpl.LoadUp(true);

			mRootWindow.SetIndex(mRootWindow.GetConversation().GetIndexLines(dpl.Guid));

			mRootWindow.ContextSwitch();
			mRootWindow.LoadInComboBoxes();

			UpdateMainConvo();
		}

		/// <summary>
		/// Copy a line to a buffer
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void CopyLine(object sender, RoutedEventArgs e)
		{
			Button b = (Button)(sender);
			DisplayLine dpl = (DisplayLine)VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(b))));

			mCopiedDisplayLine = dpl;

			dpl.CopyOut(true);
		}

		/// <summary>
		/// Update the focus from the view
		/// </summary>
		private void UpdateMainConvo()
		{
			mRootWindow.UpdateConvoFromDisplay();
		}

		/// <summary>
		/// Add a new line in the view
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void AddNewLine(object sender, MouseButtonEventArgs e)
		{
			Line newLin = mRootWindow.GetConversation().InventLine();
			if (mCopiedDisplayLine != null)
			{
				var copiedRootLine = mRootWindow.GetConversation().Lines.Find(x => x.Guid == mCopiedDisplayLine.Guid);

				// ToDo: Copy over more information than just text
				newLin.Dialogue = copiedRootLine.Dialogue;
				newLin.Speaker = copiedRootLine.Speaker;

				foreach (var input in copiedRootLine.InputDirections)
				{
					newLin.AddInputDirection(new DirectionInput(input));
				}
				
				foreach (var output in copiedRootLine.OutputDirections)
				{
					newLin.AddOutputDirection(new DirectionOutput(output));
				}

			}

			newLin.Root = mRootWindow.GetConversation().Lines.Count == 0;

			DisplayLine newDisplayLine = MakeDisplayLine(newLin);
			mDisplayLines.Add(newDisplayLine);

			MyCanvas.Children.Add(newDisplayLine);

			Point mousePos = e.GetPosition(null);

			var vo = MainScroll.VerticalOffset;
			var ho = MainScroll.HorizontalOffset;

			var mouseX = (mousePos.X / scale) + (ho / scale);
			var mouseY = (mousePos.Y / scale) + (vo / scale);

			Canvas.SetTop(newDisplayLine, mouseY);

			Canvas.SetLeft(newDisplayLine, mouseX);

			UpdateMainConvo();
			mRootWindow.UpdateUndoList();
		}

		/// <summary>
		/// Add a new edge in the view
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void AddNewEdge(object sender, RoutedEventArgs e)
		{
			Button b = (Button)(sender);
			DisplayLine dpl = (DisplayLine)VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(VisualTreeHelper.GetParent(b))));

			if (!mMultipleSelection)
			{
				if (mAddingMode)
				{
					mAddingMode = false;

					if (mTempEdgeSource != dpl.Guid)
					{
						foreach (var match in mRootWindow.GetConversation().Lines.Find(x => x.Guid == dpl.Guid).Inputs)
						{
							if (match.Source.Guid.Equals(mTempEdgeSource))
							{
								mDisplayLines.Find(x => x.Guid == mTempEdgeSource).AddEdge(false);
								return;
							}
						}

						Edge newRep = mRootWindow.GetConversation().InventEdge();
						newRep.Source = mRootWindow.GetConversation().Lines.Find(x => x.Guid == mTempEdgeSource);
						newRep.Source.Outputs.Add(newRep);
						newRep.Destination = mRootWindow.GetConversation().Lines.Find(x => x.Guid == dpl.Guid);
						newRep.Destination.Inputs.Add(newRep);
						UpdateMainConvo();

						var newDisplayEdge = MakeDisplayReply(newRep);

						mDisplayReplies.Add(newDisplayEdge);

						newDisplayEdge.SetPosition();
						newDisplayEdge.DisplaySource.AddEdge(false);

						MyCanvas.Children.Add(newDisplayEdge);
					}
					else
					{
						dpl.AddEdge(false);
					}
				}
				else
				{
					mAddingMode = true;

					mTempEdgeSource = dpl.Guid;

					dpl.AddEdge(true);
				}
			}
			mRootWindow.UpdateUndoList();

		}

		/// <summary>
		/// Update the display from main
		/// </summary>
		/// <param name="main"></param>
		public void ReloadDisplay(MainWindow main)
		{
			mRootWindow = main;
			// ToDo: optimize this without screwing up visuals?
			MyCanvas.Children.Clear();
			mDisplayLines.Clear();
			mDisplayReplies.Clear();

			mRootWindow.SetConversation(SetParents(mRootWindow.GetConversation()));

			MakeDisplayLines(mRootWindow.GetConversation(), main.GetCurrentIndex());
			PositionLines(mRootWindow.GetConversation(), FindBiggerLayer(mRootWindow.GetConversation()));
			MakeDisplayReplies(mRootWindow.GetConversation());
			PositionReplies(mRootWindow.GetConversation());

		}

		/// <summary>
		/// Key down handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void KeyPressHandler(object sender, KeyEventArgs e)
		{

			if (e.Key == Key.LeftCtrl)
			{
				mControlMode = true;
			}

			if (e.Key == Key.LeftShift || e.Key == Key.RightShift)
			{
				mMultipleSelection = true;
			}

			if (e.Key == Key.OemMinus && mControlMode)
			{
				if (scale > zoomOutMax)
					scale /= 1.1f;
				MyCanvas.LayoutTransform = new ScaleTransform(scale, scale);
			}


			if (e.Key == Key.OemPlus && mControlMode)
			{
				if (scale < zoomInMax)
					scale *= 1.1f;
				MyCanvas.LayoutTransform = new ScaleTransform(scale, scale);
			}


			if (e.Key == Key.C && mControlMode && mCopiedDisplayLine != null)
			{
				mCopiedDisplayLine.CopyOut(false);
				mCopiedDisplayLine = null;
			}
		}

		/// <summary>
		/// Key up handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void KeyReleaseHandler(object sender, KeyEventArgs e)
		{
			if (e.Key == Key.LeftCtrl)
			{
				mControlMode = false;
			}


			if (e.Key == Key.LeftShift || e.Key == Key.RightShift)
			{
				mMultipleSelection = false;
				foreach (DisplayLine dpl in mDisplayLines)
				{
					dpl.Selection(false);
				}
			}
		}

		/// <summary>
		/// Let go of the dragging element
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void Release(object sender, MouseButtonEventArgs e)
		{
			if (!mMultipleSelection)
			{
				foreach (var line in mDisplayLines)
					line.Selection(false);
			}
		}

	}
}
