﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Shapes;

namespace CanterburySharp
{
	/// <summary>
	/// Interaction logic for FullScript.xaml
	/// </summary>
	public partial class FullScript : Window
	{
		Dictionary<Button, Edge> mButtonToEdgeMap = new Dictionary<Button, Edge>();

		List<Removable> mRemovableSegments = new List<Removable>();

		Conversation mConvoCopy;

		public bool valid = false;

		/// <summary>
		/// A class to represent an element in the conversation
		/// </summary>
		class Removable
		{
			public TextBox mText;
			public List<Button> mButtons;
			public List<Direction> mSavedDirections;
			public Line mLine;

			public Removable()
			{
				mSavedDirections = new List<Direction>();
			}
		}

		/// <summary>
		/// Load in the current line after picking or starting an option
		/// </summary>
		/// <param name="line"></param>
		public void LoadUp(Line line)
		{
			ApplyLogic(line);

			TextBox text = new TextBox();
			text.Text = line.Speaker;
			text.Text += ":\n" + line.Dialogue;

			MyStackPanel.Children.Add(text);

			Removable cur = new Removable();
			cur.mLine = line;
			cur = SaveDirections(cur);
			cur.mText = text;
			cur.mText.IsEnabled = false;

			cur.mButtons = new List<Button>();

			int count = 0;
			// ToDo: not just 0th index, whatever the starting is
			foreach (var reply in line.Outputs)
			{
				Button but = new Button();
				but.Content = ((Line)reply.Destination).Full;

				but.Click += EdgePicked;

				MyStackPanel.Children.Add(but);
				mButtonToEdgeMap.Add(but, (Edge)reply);

				cur.mButtons.Add(but);

				but.IsEnabled = ValidateLogic((Line)reply.Destination);

				count++;
			}

			mRemovableSegments.Add(cur);
		}
	
		public FullScript(Conversation convo, int index = 0)
		{
			mConvoCopy = new Conversation(convo);

			InitializeComponent();


			foreach (Line line in mConvoCopy.Lines)
			{
				line.ParentChain = -1;
			}

			foreach (var line in mConvoCopy.Lines)
			{
				line.FindParentagePosition();
			}

			var entries = mConvoCopy.Lines.FindAll(x => x.Root == true);
			
			if (entries.Count <= index || index == -1)
			{
				MessageBox.Show("Could not find an appropriate entry point!");
				valid = false;
				return;
			}
			Line starter = entries[index];

			LoadUp(starter);

			foreach (var direction in mConvoCopy.StageDirections)
			{
				StageDirection sd = new StageDirection();
				DirectionPanel.Children.Add(sd);
				sd.DataContext = direction;
				sd.DirectionName.IsEnabled = false;
				sd.Delete.IsEnabled = false;
				sd.Value.IsEnabled = true;
			}

			valid = true;

		}
		
		/// <summary>
		/// Option has been selected by the user
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void EdgePicked(object sender, RoutedEventArgs e)
		{
			foreach (var rem in mRemovableSegments.Last().mButtons)
			{
				rem.IsEnabled = false;
			}

			Edge value;
			mButtonToEdgeMap.TryGetValue(((Button)sender), out value);

			LoadUp(((Line)(value.Destination)));

			Back.IsEnabled = true;

			Scroller.ScrollToVerticalOffset(double.PositiveInfinity);

		}

		/// <summary>
		/// Revert to previous conversation state
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void GoBack(object sender, RoutedEventArgs e)
		{
			if (mRemovableSegments.Count == 0)
			{
				return;
			}

			MyStackPanel.Children.Remove(mRemovableSegments.Last().mText);
			foreach (var each in mRemovableSegments.Last().mButtons)
				MyStackPanel.Children.Remove(each);

			mRemovableSegments.Remove(mRemovableSegments.Last());

			if (mRemovableSegments.Count == 1)
			{
				Back.IsEnabled = false;
			}

			for (int i = 0; i < mConvoCopy.StageDirections.Count; i++)
			{
				mConvoCopy.StageDirections[i].DirectionValue = mRemovableSegments.Last().mSavedDirections.Find(x => x.DirectionName.Equals(mConvoCopy.StageDirections[i].DirectionName)).DirectionValue;
			}

			int counter = 0;
			foreach (var each in mRemovableSegments.Last().mButtons)
			{
				each.IsEnabled = ValidateLogic((Line)mRemovableSegments.Last().mLine.Outputs[counter].Destination);
				counter++;
			}
		}

		/// <summary>
		/// Check if an output line is allowed with the current world state.
		/// </summary>
		/// <param name="line"></param>
		/// <returns></returns>
		private bool ValidateLogic(Line line)
		{
			if (line.InputDirections.Count == 0)
				return true;

			bool checks = true;

			foreach (var and in line.InputDirections.FindAll(x => x.LogicValue == DirectionInput.Logic.And))
			{
				var myDir = mConvoCopy.StageDirections.Find(x => x.DirectionName == and.DirectionName);

				double myDirDoub, inputDoub;
				switch (and.ComparisonValue)
				{
					case DirectionInput.Comparison.Equal:
						checks = myDir.DirectionValue == and.DirectionValue;
						break;
					case DirectionInput.Comparison.LessThan:
						if (!Double.TryParse(myDir.DirectionValue, out myDirDoub) || !Double.TryParse(and.DirectionValue, out inputDoub))
							break;
						checks = myDirDoub < inputDoub;
						break;
					case DirectionInput.Comparison.MoreThan:
						if (!Double.TryParse(myDir.DirectionValue, out myDirDoub) || !Double.TryParse(and.DirectionValue, out inputDoub))
							break;
						checks = myDirDoub > inputDoub;
						break;
					case DirectionInput.Comparison.NotEqual:
						checks = myDir.DirectionValue != and.DirectionValue;
						break;

					case DirectionInput.Comparison.Contains:
						checks = myDir.DirectionValue.Contains(and.DirectionValue);
						break;

					case DirectionInput.Comparison.DoesNotContain:
						checks = !myDir.DirectionValue.Contains(and.DirectionValue);
						break;
				}

				if (checks == false)
					return checks;

			}

			foreach (var or in line.InputDirections.FindAll(x => x.LogicValue == DirectionInput.Logic.Or))
			{
				var myDir = mConvoCopy.StageDirections.Find(x => x.DirectionName == or.DirectionName);

				checks = false;

				double myDirDoub, inputDoub;
				switch (or.ComparisonValue)
				{
					case DirectionInput.Comparison.Equal:
						checks = myDir.DirectionValue == or.DirectionValue;
						break;
					case DirectionInput.Comparison.LessThan:
						if (!Double.TryParse(myDir.DirectionValue, out myDirDoub) || !Double.TryParse(or.DirectionValue, out inputDoub))
							break;
						checks = myDirDoub < inputDoub;
						break;
					case DirectionInput.Comparison.MoreThan:
						if (!Double.TryParse(myDir.DirectionValue, out myDirDoub) || !Double.TryParse(or.DirectionValue, out inputDoub))
							break;
						checks = myDirDoub > inputDoub;
						break;
					case DirectionInput.Comparison.NotEqual:
						checks = myDir.DirectionValue != or.DirectionValue;
						break;

					case DirectionInput.Comparison.Contains:
						checks = myDir.DirectionValue.Contains(or.DirectionValue);
						break;

					case DirectionInput.Comparison.DoesNotContain:
						checks = !myDir.DirectionValue.Contains(or.DirectionValue);
						break;
				}

				if (checks == true)
					return checks;
			}

			return checks;

		}
		
		/// <summary>
		/// Applies output directions of a line
		/// </summary>
		/// <param name="line"></param>
		private void ApplyLogic(Line line)
		{
			foreach (var output in line.OutputDirections)
			{
				var myDir = mConvoCopy.StageDirections.Find(x => x.DirectionName == output.DirectionName);

				double myDirDoub = 0, outputDoub = 0;
				switch (output.AssignmentValue)
				{
					case DirectionOutput.Assignment.Equals:
						myDir.DirectionValue = output.DirectionValue;
						break;
					case DirectionOutput.Assignment.MinusEquals:
						if (!Double.TryParse(myDir.DirectionValue, out myDirDoub) || !Double.TryParse(output.DirectionValue, out outputDoub))
							break;
						myDir.DirectionValue = (myDirDoub -= outputDoub).ToString();
						break;
					case DirectionOutput.Assignment.PlusEquals:
						if (!Double.TryParse(myDir.DirectionValue, out myDirDoub) || !Double.TryParse(output.DirectionValue, out outputDoub))
							break;
						myDir.DirectionValue = (myDirDoub += outputDoub).ToString();
						break;
				}
			}
		}

		/// <summary>
		/// Save off a snapshot of our directions.
		/// </summary>
		/// <param name="rem"></param>
		/// <returns></returns>
		private Removable SaveDirections(Removable rem)
		{
			rem.mSavedDirections.Clear();
			foreach (var dir in mConvoCopy.StageDirections)
			{
				rem.mSavedDirections.Add(new Direction(dir));
			}

			return rem;
		}

	}
}
