﻿using Newtonsoft.Json;
using SharpGL;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using System.Xml;

namespace CanterburySharp
{
	/// <summary>
	/// Interaction logic for MainWindow.xaml
	/// </summary>
	public partial class MainWindow : Window
	{
		public Story MainStory = new Story();

		internal bool Autosave = false;

		public Stack<Conversation> PastVersions = new Stack<Conversation>();
		public Stack<Conversation> FutureVersions = new Stack<Conversation>();

		public StoryPage storyPage;
		public FocusPage focusPage;
		public DisplayPage displayPage;
		public DirectionPage directionPage;

		bool heldControl;

		bool dark = false;

		public bool Undoing { get; internal set; }

		/// <summary>
		/// Gets the current conversation from the main page.
		/// </summary>
		/// <returns></returns>
		public Conversation GetConversation()
		{
			return storyPage.GetConversation();
		}

		/// <summary>
		/// Called during autosave
		/// </summary>
		public void Save()
		{
			if (Autosave && storyPage.LocalFileName.Length > 0)
			{
				FileInfo f = new FileInfo(storyPage.AutoFilePath);
				if (File.Exists(storyPage.AutoFilePath) &&
					f.IsReadOnly)
				{
					MessageBox.Show("Hey dingus, that file is read only! Pick a different name or change the properties!");
					return;
				}

				File.WriteAllText(storyPage.AutoFilePath, JsonConvert.SerializeObject(MainStory));
			}
		}

		/// <summary>
		/// Updates the undo list.
		/// </summary>
		public void UpdateUndoList()
		{
			if (Undoing)
				return;

			Save();

			if (!PastVersions.Peek().AreSame(storyPage.GetConversation()))
			{
				PastVersions.Push(new Conversation(storyPage.GetConversation()));
				FutureVersions.Clear();
			}

			UndoOption.IsEnabled = PastVersions.Count > 1;
			RedoOption.IsEnabled = FutureVersions.Count != 0;
		}

		/// <summary>
		/// Revert to previous conversation
		/// </summary>
		void Undo()
		{
			if (PastVersions.Count < 2)
				return;

			Undoing = true;

			FutureVersions.Push(new Conversation(storyPage.GetConversation()));
			PastVersions.Pop();
			storyPage.SetConversation(new Conversation(PastVersions.Peek()));

			UndoOption.IsEnabled = PastVersions.Count > 1;
			RedoOption.IsEnabled = FutureVersions.Count != 0;

			focusPage.CurrentIndex = Math.Min(GetConversation().Lines.Count - 1, focusPage.CurrentIndex);
			focusPage.DataContext = GetConversation().Lines[focusPage.CurrentIndex];
			
			displayPage.ReloadDisplay(this);
			focusPage.LoadInComboBoxes();
			focusPage.ContextSwitch();

			Undoing = false;
		}

		/// <summary>
		/// Go to future version
		/// </summary>
		void Redo()
		{
			if (FutureVersions.Count == 0)
				return;

			Undoing = true;

			PastVersions.Push(new Conversation(storyPage.GetConversation()));
			storyPage.SetConversation(new Conversation(FutureVersions.Pop()));
						
			UndoOption.IsEnabled = PastVersions.Count > 1;
			RedoOption.IsEnabled = FutureVersions.Count != 0;

			focusPage.CurrentIndex = Math.Min(GetConversation().Lines.Count - 1, focusPage.CurrentIndex);
			focusPage.DataContext = GetConversation().Lines[focusPage.CurrentIndex];

			displayPage.ReloadDisplay(this);
			focusPage.LoadInComboBoxes();
			focusPage.ContextSwitch();
			
			Undoing = false;
		}

		/// <summary>
		/// Reload combo boxes for the stage directions.
		/// </summary>
		internal void LoadStageDirectionCombos()
		{
			focusPage.LoadStageDirectionCombos();
		}

		/// <summary>
		/// Constructor
		/// </summary>
		public MainWindow()
		{
			storyPage = new StoryPage(this);

			InitializeComponent();
			StoryFrame.Content = storyPage;

			
			FocusTab.IsEnabled = false;
			DisplayTab.IsEnabled = false;
			DirectionTab.IsEnabled = false;
		}

		/// <summary>
		/// Enables the tabs if we have a valid conversation selected
		/// </summary>
		/// <param name="enabled"></param>
		public void EnableOtherTabs(bool enabled)
		{
			FocusFrame.Content = null;
			DisplayFrame.Content = null;
			DirectionFrame.Content = null;;

			if (enabled)
			{
				focusPage = new FocusPage(this);
				displayPage = new DisplayPage(this);
				directionPage = new DirectionPage(this);

				UndoOption.IsEnabled = false;
				RedoOption.IsEnabled = false;

				FutureVersions.Clear();
				PastVersions.Clear();

				PastVersions.Push(new Conversation(storyPage.GetConversation()));

			}
			else
			{
				focusPage = null;
				displayPage = null;
				directionPage = null;


				UndoOption.IsEnabled = false;
				RedoOption.IsEnabled = false;

				FutureVersions.Clear();
				PastVersions.Clear();

			}

			FocusFrame.Content = focusPage;
			DisplayFrame.Content = displayPage;
			DirectionFrame.Content = directionPage;

			FocusTab.IsEnabled = enabled;
			DisplayTab.IsEnabled = enabled;
			DirectionTab.IsEnabled = enabled;
		}

		/// <summary>
		/// Open from a file.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void OpenStory(object sender, RoutedEventArgs e)
		{
			System.Windows.Forms.OpenFileDialog openFileDialog = new System.Windows.Forms.OpenFileDialog();
			openFileDialog.Filter = "json files (*.json)|*.json";
			if (openFileDialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
			{
				string read = File.ReadAllText(openFileDialog.FileName);
				if (!String.IsNullOrEmpty(read))
				{
					if (MessageBox.Show("This will overwrite whatever you're working on. Are you sure you want to continue?", "Question", MessageBoxButton.YesNo, MessageBoxImage.Question) == MessageBoxResult.No)
					{
						return;
					}

					try
					{
						MainStory = JsonConvert.DeserializeObject<Story>(read);
						DataContext = MainStory;
						storyPage.DataContext = MainStory;
						string str = openFileDialog.FileName.Split('\\').Last();
						storyPage.LocalFileName = str.Substring(0, str.Length - 5);
						storyPage.AutoFilePath = openFileDialog.FileName;
						storyPage.SendEmail.IsEnabled = true;
					}
					catch
					{
						MessageBox.Show("Something went wrong with deserialization. Don't load that file again, I guess?");
						return;
					}

					storyPage.ResetSelectors();
					storyPage.LoadActs();

					MessageBox.Show("Loaded in story \"" + MainStory.Name + "\" correctly.");
				}
			}
		}

		/// <summary>
		/// Save the story to a file.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		internal void SaveStory(object sender, RoutedEventArgs e)
		{
			System.Windows.Forms.SaveFileDialog saveFileDialog = new System.Windows.Forms.SaveFileDialog();
			saveFileDialog.Filter = "json files (*.json)|*.json";
			if (saveFileDialog.ShowDialog() == System.Windows.Forms.DialogResult.OK)
			{
				string str = saveFileDialog.FileName.Split('\\').Last();
				storyPage.LocalFileName = str.Substring(0, str.Length - 5);
				storyPage.AutoFilePath = saveFileDialog.FileName;
				File.WriteAllText(saveFileDialog.FileName, JsonConvert.SerializeObject(MainStory));
				storyPage.SendEmail.IsEnabled = true;
			}
		}

		/// <summary>
		/// Close the application
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void Close(object sender, RoutedEventArgs e)
		{
			Save();
			Close();
		}

		/// <summary>
		/// Set the current conversation
		/// </summary>
		/// <param name="conversation"></param>
		internal void SetConversation(Conversation conversation)
		{
			storyPage.SetConversation(conversation);
		}

		/// <summary>
		/// Load in the combo boxes in the focus page.
		/// </summary>
		internal void LoadInComboBoxes()
		{
			focusPage.LoadInComboBoxes();
		}

		/// <summary>
		/// Update every conversation
		/// </summary>
		internal void UpdateConvoFromDisplay()
		{
			focusPage.UpdateConvoFromDisplay();
		}

		/// <summary>
		/// Changes the focused conversation.
		/// </summary>
		internal void ContextSwitch()
		{
			focusPage.ContextSwitch();
		}

		/// <summary>
		/// Set the index of the focus page.
		/// </summary>
		/// <param name="newIndex"></param>
		internal void SetIndex(int newIndex)
		{
			focusPage.CurrentIndex = newIndex;
		}

		/// <summary>
		/// Reload the display view.
		/// </summary>
		internal void ReloadDisplay()
		{
			if (displayPage != null)

			displayPage.ReloadDisplay(this);
		}

		/// <summary>
		/// Get the index in the focus page.
		/// </summary>
		/// <returns></returns>
		public int GetCurrentIndex()
		{
			return focusPage.CurrentIndex;
		}

		/// <summary>
		/// Undo from the drop 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void UndoOption_Click(object sender, RoutedEventArgs e)
		{
			Undo();
		}

		/// <summary>
		/// 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void RedoOption_Click(object sender, RoutedEventArgs e)
		{
			Redo();
		}

		/// <summary>
		/// Key down handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void Tabs_KeyDown(object sender, KeyEventArgs e)
		{
			if (e.Key == Key.LeftCtrl || e.Key == Key.RightCtrl)
			{
				heldControl = true;
			}

			if (e.Key == Key.Z && heldControl)
			{
				Undo();
			}

			if (e.Key == Key.Y && heldControl)
			{
				Redo();
			}
		}

		/// <summary>
		/// Key release handler
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void Tabs_KeyUp(object sender, KeyEventArgs e)
		{
			if (e.Key == Key.LeftCtrl || e.Key == Key.RightCtrl)
			{
				heldControl = false;
			}
		}

		/// <summary>
		/// toggle dark theme
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void StyleToggle(object sender, RoutedEventArgs e)
		{
			Uri style;
			if (!dark)
			{
				style = new Uri(@"StyleZone.xaml", UriKind.Relative);
			}
			else
			{
				style = new Uri(@"BadStyle.xaml", UriKind.Relative);
			}
			Application.Current.Resources.Source = style;
			dark = !dark;
			ReloadDisplay();
		}

		/// <summary>
		/// Show about window
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void About(object sender, RoutedEventArgs e)
		{
			new AboutWindow().Show();
		}

		/// <summary>
		/// Show control window.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void Control(object sender, RoutedEventArgs e)
		{
			new Controls().Show();
		}
	}
}

