﻿using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Net.Mail;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;
using Outlook = Microsoft.Office.Interop.Outlook;

namespace CanterburySharp
{
	/// <summary>
	/// Interaction logic for StoryPage.xaml
	/// </summary>
	public partial class StoryPage : Page
	{
		MainWindow mRootWindow;

		public string LocalFileName;
		public string AutoFilePath;

		Perforce.P4.Server server;
		Perforce.P4.Repository rep;
		Perforce.P4.Connection con;

		/// <summary>
		/// Conversation
		/// </summary>
		/// <param name="main"></param>
		public StoryPage(MainWindow main)
		{
			InitializeComponent();
			mRootWindow = main;
			DataContext = mRootWindow.MainStory;
		}

		/// <summary>
		/// Load act 
		/// </summary>
		public void LoadActs()
		{
			List<string> strings = new List<string>();
			foreach (var act in mRootWindow.MainStory.Acts)
			{
				strings.Add(act.Name);
			}
			ActSelect.ItemsSource = strings;
			ActSelect.SelectedIndex = strings.Count - 1;
		}

		/// <summary>
		/// Load scene
		/// </summary>
		private void LoadScenes()
		{
			if (ActSelect.SelectedIndex == -1)
			{
				return;
			}

			List<string> strings = new List<string>();
			foreach (var scene in mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes)
			{
				strings.Add(scene.Name);
			}
			SceneSelect.ItemsSource = strings;
			SceneSelect.SelectedIndex = strings.Count - 1;
		}

		/// <summary>
		/// Load conversation
		/// </summary>
		private void LoadConversations()
		{
			if (ActSelect.SelectedIndex == -1 || SceneSelect.SelectedIndex == -1)
			{
				return;
			}
			List<string> strings = new List<string>();
			foreach (var conversation in mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations)
			{
				strings.Add(conversation.Name);
			}
			ConversationSelect.ItemsSource = strings;
			ConversationSelect.SelectedIndex = strings.Count - 1;
		}

		/// <summary>
		/// Delete act
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ActDelete_Click(object sender, RoutedEventArgs e)
		{
			if (ActSelect.SelectedIndex == -1)
			{
				MessageBox.Show("No active act.");
				return;
			}
			mRootWindow.MainStory.Acts.RemoveAt(ActSelect.SelectedIndex);
			LoadActs();
			SceneSelect.ItemsSource = null;
			ConversationSelect.ItemsSource = null;

		}

		/// <summary>
		/// Delete a scene
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void SceneDelete_Click(object sender, RoutedEventArgs e)
		{
			if (ActSelect.SelectedIndex == -1 || SceneSelect.SelectedIndex == -1)
			{
				MessageBox.Show("No active scene.");
				return;
			}
			mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes.RemoveAt(SceneSelect.SelectedIndex);
			LoadScenes();
			ConversationSelect.ItemsSource = null;
		}

		/// <summary>
		/// Delete a conversation
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ConversationDelete_Click(object sender, RoutedEventArgs e)
		{
			if (ActSelect.SelectedIndex == -1 || SceneSelect.SelectedIndex == -1 || ConversationSelect.SelectedIndex == -1)
			{
				MessageBox.Show("No active conversation.");
				return;
			}
			mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations.RemoveAt(ConversationSelect.SelectedIndex);
			LoadConversations();
		}

		/// <summary>
		/// Add a new act
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ActAdd_Click(object sender, RoutedEventArgs e)
		{
			if (!String.IsNullOrEmpty(ActName.Text))
			{
				foreach (var act in mRootWindow.MainStory.Acts)
				{
					if (ActName.Text.Equals(act.Name))
					{
						MessageBox.Show("Duplicate name for this act.");
						return;
					}
				}
				mRootWindow.MainStory.AddNewAct(ActName.Text);
				LoadActs();
				SceneSelect.ItemsSource = null;
				ConversationSelect.ItemsSource = null;
			}
			else
			{
				MessageBox.Show("No act name!");
				return;
			}
		}

		/// <summary>
		/// Add a new scene
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void SceneAdd_Click(object sender, RoutedEventArgs e)
		{
			if (ActSelect.SelectedIndex == -1)
			{
				MessageBox.Show("No active act.");
				return;
			}
			if (!String.IsNullOrEmpty(SceneName.Text))
			{
				foreach (var scene in mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes)
				{
					if (SceneName.Text.Equals(scene.Name))
					{
						MessageBox.Show("Duplicate name for this scene.");
						return;
					}
				}
				mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].AddNewScene(SceneName.Text);
				LoadScenes();
				ConversationSelect.ItemsSource = null;
			}
			else
			{
				MessageBox.Show("No scene name!");
				return;
			}
		}

		public void ResetSelectors()
		{
			ActSelect.ItemsSource = null;
			SceneSelect.ItemsSource = null;
			ConversationSelect.ItemsSource = null;
		}

		/// <summary>
		/// Add a new conversation
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ConversationAdd_Click(object sender, RoutedEventArgs e)
		{
			if (ActSelect.SelectedIndex == -1 || SceneSelect.SelectedIndex == -1)
			{
				MessageBox.Show("No active scene.");
				return;
			}
			if (!String.IsNullOrEmpty(ConversationName.Text))
			{
				foreach (var conversation in mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations)
				{
					if (ConversationName.Text.Equals(conversation.Name))
					{
						MessageBox.Show("Duplicate name for this conversation.");
						return;
					}
				}
				mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].AddNewConversation(ConversationName.Text);
				LoadConversations();
			}
			else
			{
				MessageBox.Show("No conversation name!");
				return;
			}
		}

		/// <summary>
		/// Change the current act
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ActSelect_SelectionChanged(object sender, SelectionChangedEventArgs e)
		{
			LoadScenes();
			LoadConversations();
		}

		/// <summary>
		/// Change the current scene 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void SceneSelect_SelectionChanged(object sender, SelectionChangedEventArgs e)
		{
			LoadConversations();
		}

		/// <summary>
		/// Change the current conversation
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ConversationSelect_SelectionChanged(object sender, SelectionChangedEventArgs e)
		{
			mRootWindow.EnableOtherTabs(ConversationSelect.SelectedIndex != -1);
		}

		/// <summary>
		/// Get the current conversation
		/// </summary>
		/// <returns></returns>
		public Conversation GetConversation()
		{
			if (ActSelect.SelectedIndex == -1 || SceneSelect.SelectedIndex == -1 || ConversationSelect.SelectedIndex == -1)
				return null;
			return mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations[ConversationSelect.SelectedIndex];
		}

		/// <summary>
		/// Set the current conversation's value
		/// </summary>
		/// <param name="conversation"></param>
		public void SetConversation(Conversation conversation)
		{
			if (ActSelect.SelectedIndex == -1 || SceneSelect.SelectedIndex == -1 || ConversationSelect.SelectedIndex == -1)
				return;
			mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations[ConversationSelect.SelectedIndex] = conversation;
		}

		/// <summary>
		/// Enable autosave
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void SetAutosaveCheck(object sender, RoutedEventArgs e)
		{
			var check = (CheckBox)(sender);
			mRootWindow.Autosave = (check.IsChecked != null && check.IsChecked == true);
			if (String.IsNullOrEmpty(LocalFileName))
			{
				mRootWindow.SaveStory(null, null);
			}
		}

		/// <summary>
		/// Rename the selected act
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void RenameAct_Click(object sender, RoutedEventArgs e)
		{
			if (ActSelect.SelectedIndex != -1 && ActName.Text.Length > 0 && mRootWindow.MainStory.Acts.FindAll(x => x.Name.Equals(ActName.Text)).Count == 0)
			{
				mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Name = ActName.Text;
				LoadActs();
			}
		}

		/// <summary>
		/// Rename the selected scene
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void RenameScene_Click(object sender, RoutedEventArgs e)
		{
			if (SceneSelect.SelectedIndex != -1 && SceneName.Text.Length > 0 && mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes.FindAll(x => x.Name.Equals(SceneName.Text)).Count == 0)
			{
				mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Name = SceneName.Text;
				LoadScenes();
			}
		}

		/// <summary>
		/// Rename conversation
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void RenameConversation_Click(object sender, RoutedEventArgs e)
		{
			if (ConversationSelect.SelectedIndex != -1 && ConversationName.Text.Length > 0 && mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations.FindAll(x => x.Name.Equals(ConversationName.Text)).Count == 0)
			{
				mRootWindow.MainStory.Acts[ActSelect.SelectedIndex].Scenes[SceneSelect.SelectedIndex].Conversations[ConversationSelect.SelectedIndex].Name = ConversationName.Text;
				LoadConversations();
			}
		}

		/// <summary>
		/// Open outlook and SEND that email
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ButtonSendEmail(object sender, RoutedEventArgs e)
		{
			if (string.IsNullOrEmpty(LocalFileName))
				return;
			Outlook.Application oApp = new Outlook.Application();
			Outlook.MailItem oMsg = (Outlook.MailItem)oApp.CreateItem(Outlook.OlItemType.olMailItem);

			oMsg.To = Address.Text;
			oMsg.Subject = Subject.Text;
			oMsg.HTMLBody = Body.Text;
			oMsg.Attachments.Add(System.IO.Path.GetFullPath(LocalFileName + ".json"));
			oMsg.Display(true); //In order to display it in modal inspector change the argument to true

		}

		/// <summary>
		/// Connect to perforce initially
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void ConnectPerforce(object sender, RoutedEventArgs e)
		{
			string uri = Server.Text;
			string user = User.Text;
			string ws_client = Workspace.Text;
			string pass = Password.Password;

			server = new Perforce.P4.Server(new Perforce.P4.ServerAddress(uri));
			rep = new Perforce.P4.Repository(server);
			con = rep.Connection;

			con.UserName = user;
			con.Client = new Perforce.P4.Client();
			con.Client.Name = ws_client;
			try
			{
				con.Connect(null);

				Perforce.P4.Credential cred = con.Login(pass, null, null);

				Perforce.P4.ServerMetaData p4info = rep.GetServerMetaData(null);
				Perforce.P4.ServerVersion version = p4info.Version;
				string release = version.Major;
				SetConnection(con == null || con.Status == Perforce.P4.ConnectionStatus.Disconnected);

			}
			catch
			{

				MessageBox.Show("Connection failed!");
			}
		}

		/// <summary>
		/// Check out from perforce
		/// </summary>
		void EditPerforce()
		{
			if (con == null || con.Status == Perforce.P4.ConnectionStatus.Disconnected || String.IsNullOrEmpty(Path.Text) || String.IsNullOrEmpty(LocalFileName))
			{
				return;
			}
			string depotPath = Path.Text + LocalFileName + ".json";
			int changelist = 0;
			Perforce.P4.FileSpec fileToCheckout = new Perforce.P4.FileSpec(new Perforce.P4.DepotPath(depotPath), null, null, null);
			Perforce.P4.EditCmdOptions options = new Perforce.P4.EditCmdOptions(Perforce.P4.EditFilesCmdFlags.None, changelist, null);

			try
			{
				IList<Perforce.P4.FileSpec> filesCheckedOut = con.Client.EditFiles(options, fileToCheckout);

			}
			catch
			{
				MessageBox.Show("Edit failed!");
			}
		}

		/// <summary>
		/// Check in to perforce
		/// </summary>
		void SubmitPerforce()
		{
			if (con == null || con.Status == Perforce.P4.ConnectionStatus.Disconnected || String.IsNullOrEmpty(Path.Text) || String.IsNullOrEmpty(LocalFileName))
			{
				return;
			}
			string description = "updated from Canterbury!";
			Perforce.P4.ClientSubmitOptions clientOptions = new Perforce.P4.ClientSubmitOptions(false, Perforce.P4.SubmitType.SubmitUnchanged);
			Perforce.P4.SubmitCmdOptions options = new Perforce.P4.SubmitCmdOptions(Perforce.P4.SubmitFilesCmdFlags.None, 0, null, description, clientOptions);
			string depotPath = Path.Text + LocalFileName + ".json";
			Perforce.P4.FileSpec fileToSubmit = new Perforce.P4.FileSpec(new Perforce.P4.DepotPath(depotPath), null, null, null);

			try
			{
				Perforce.P4.SubmitResults results = con.Client.SubmitFiles(options, fileToSubmit);
			}
			catch
			{
				MessageBox.Show("Submit failed!");
			}
		}

		/// <summary>
		/// Disconnect from perforce
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DisconnectPerforce(object sender, RoutedEventArgs e)
		{
			if (con != null)
				con.Disconnect();
			SetConnection(con == null || con.Status == Perforce.P4.ConnectionStatus.Disconnected);
		}

		private void EditClick(object sender, RoutedEventArgs e)
		{
			EditPerforce();
			SetConnection(con == null || con.Status == Perforce.P4.ConnectionStatus.Disconnected);
		}

		private void SubmitClick(object sender, RoutedEventArgs e)
		{
			SubmitPerforce();
			SetConnection(con == null || con.Status == Perforce.P4.ConnectionStatus.Disconnected);
		}

		/// <summary>
		/// Change our buttons now that we're connected or disconnected
		/// </summary>
		/// <param name="instructions"></param>
		void SetConnection(bool instructions)
		{
			p4Check.IsEnabled = !instructions;
			p4Connect.IsEnabled = instructions;
			p4Disconnect.IsEnabled = !instructions;
			p4Submit.IsEnabled = !instructions;
		}
	}
}
