﻿using Newtonsoft.Json;
using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace CanterburySharp
{
	public class Vertex : NotifyableObject
	{

		private List<Edge> mInputs = new List<Edge>();
		private List<Edge> mOutputs = new List<Edge>();

		private int mHorizontalPosition = 0;
		private int mVerticalPosition = 0;

		private int mParentChain = -1;

		private bool mRoot = false;

		private Guid mGuid = Guid.NewGuid();

		/// <summary>
		/// Is this a valid entry point?
		/// </summary>
		public bool Root
		{
			get => mRoot; set
			{
				mRoot = value;
				RaisePropertyChange("Root");
				RaisePropertyChange("NotRoot");
			}
		}

		public Guid Guid { get => mGuid; set => mGuid = value; }

		/// <summary>
		/// All edges that lead to this node
		/// </summary>
		[JsonIgnore]
		public List<Edge> Inputs { get => mInputs; set { mInputs = value;
				RaisePropertyChange("Inputs");
			}
		}

		/// <summary>
		/// All edges that lead from this node.
		/// </summary>
		[JsonIgnore]
		public List<Edge> Outputs { get => mOutputs; set
			{
				mOutputs = value;
				RaisePropertyChange("Outputs");
			}
		}

		/// <summary>
		/// Numeric representation of where you stand numerically in the view.
		/// </summary>
		public int ParentChain {
			get {
				if (mRoot)
					return 0;
				else
					return mParentChain;
			} set => mParentChain = value; }

		/// <summary>
		/// Helper func with a tracker list.
		/// </summary>
		/// <returns></returns>
		public int FindParentagePosition()
		{
			return FindParentagePosition(new List<Guid>());
		}

		/// <summary>
		/// Trace upward iteratively to find out where to place in the view.
		/// </summary>
		/// <param name="visited"></param>
		/// <returns></returns>
		public int FindParentagePosition(List<Guid> visited)
		{
			if (!visited.Contains(mGuid))
			{
				visited.Add(mGuid);
			}
			else
				return mParentChain;

			if (Inputs.Count == 0 || mRoot)
			{
				mParentChain = 0;
			}
			{
				if (Inputs.Count > 0 && !mRoot)
				{
					mParentChain = short.MaxValue;
					foreach (var input in Inputs)
					{
						if (input == null || input.Source == null || !visited.Contains(input.Source.Guid))
							continue;
						mParentChain = Math.Min(mParentChain, input.Source.mParentChain + 1);
					}
				}
			}

			foreach (var output in Outputs)
			{
				if (!visited.Contains(output.Destination.Guid))
					output.Destination.FindParentagePosition(visited);
			}

			return mParentChain;
		}

		/// <summary>
		/// Add an input edge.
		/// </summary>
		/// <param name="toAdd"></param>
		/// <returns></returns>
		public bool AddInputEdge(Edge toAdd)
		{
			if (!mInputs.Contains(toAdd))
			{
				mInputs.Add(toAdd);
				RaisePropertyChange("Inputs");
				return true;
			}
			RaisePropertyChange("Inputs");

			return false;
		}

		/// <summary>
		/// Add an output edge.
		/// </summary>
		/// <param name="toAdd"></param>
		/// <returns></returns>
		public bool AddOutputEdge(Edge toAdd)
		{
			if (!mOutputs.Contains(toAdd))
			{
				mOutputs.Add(toAdd);
				RaisePropertyChange("Outputs");
				return true;
			}

			RaisePropertyChange("Outputs");
			return false;
		}

		/// <summary>
		/// Remove an input edge.
		/// </summary>
		/// <param name="toRemove"></param>
		/// <returns></returns>
		public bool RemoveInputEdge(Edge toRemove)
		{
			if (mInputs.Contains(toRemove))
			{
				mInputs.Remove(toRemove);
				RaisePropertyChange("Inputs");
				return true;
			}

			return false;
		}

		/// <summary>
		/// Remove an output edge. 
		/// </summary>
		/// <param name="toRemove"></param>
		/// <returns></returns>
		public bool RemoveOutputEdge(Edge toRemove)
		{
			if (mOutputs.Contains(toRemove))
			{
				mOutputs.Remove(toRemove);
				RaisePropertyChange("Outputs");
				return true;
			}
			RaisePropertyChange("Outputs");

			return false;
		}

	}
}
