*-----------------------------------------------------------
* Title      :  DataChunkReader
* Written by :  Josh Wade
* Date       :  10/21/2018
* Description:  Part two of a two part assignment
*               Reads in generated data
*               Each section has a random number of bytes, shorts, and longs
*               Each one of these bytes, shorts, and longs has a random value
*               Runs calculations on this data to validate stats file 
*-----------------------------------------------------------
    ORG    $1000

; EQUs

; Masks the lower word
; Useful for divus
BOTTOM_WORD_MASK    EQU     $FFFF

; Values used for determining what point to round to 
ROUNDING_POINT_PLUS_1       EQU     1000
ROUNDING_POINT              EQU     100

; Different sizes for the data types we'll be working with 
SIZE_BYTE   EQU 1
SIZE_WORD   EQU 2
SIZE_LONG   EQU 4

; How many chunks are in every section
CHUNKS_PER_SECTION  EQU 3

; Trap code values
TASK_PRINT_NUMBER                   EQU 3
TASK_READ_INPUT                     EQU 4
TASK_PRINT_MESSAGE_WITH_CRLF        EQU 13
TASK_PRINT_MESSAGE_WITHOUT_CRLF     EQU 14

; Every section will be this big, in bytes
; Max size for byte chunk = 500 = 500 bytes
; Max size for short chunk = 500 = 1000 bytes
; Max size for long chunk = 500 = 2000 bytes
CONSTANT_SECTION_SIZE       EQU 3500
CONSTANT_BYTE_CHUNK_SIZE    EQU 500
CONSTANT_WORD_CHUNK_SIZE    EQU 1000
CONSTANT_LONG_CHUNK_SIZE    EQU 2000


**  BIN FILE AND METADATA FUNCTIONS  **


; Loads in the bin file and saves metadata from its header
LoadBinFile
    ; a0 points to the binary file
    lea binary_file, a0
    
    ; Check if this file is formatted correctly
    move.l  (id_field), d0
    cmp.l   (a0)+, d0
    beq     ContinueLoad
    
    ; Throw error if it isn't
    jsr     ErrorMessage   
    rts 
        
ContinueLoad
    ; Save the number of sections
    move.l  a0, a1
    move.l  (a1)+, d0
    move.l  d0, (number_of_sections)
    
    ; Calculate the chunk offset to figure out where the data starts 
    ; Take the number of sections, multiply it by the chunks per section, and multiply it by the the size of the chunk size number (4)
    ; The result is the end of the header section
    lsl.l   #2, d0
    mulu.w  #CHUNKS_PER_SECTION, d0
    add.l   d0, a1
    
    ; Save the header offset
    move.l  a1, (data_sections_start)
    
    rts


; Validates that the section entered exists
ValidateSection
    ; See if the player entered in a cheeky/invalid amount of sections
    ; First, if the entry is 0 / negative
    cmp.l   #0, d1
    bgt     CompareMax
    jsr     OOBMessage
    rts
    
CompareMax
    ; Next, if it's beyond our section count
    cmp.l   (number_of_sections), d1
    ble     SafeNumber   
    jsr     OOBMessage
    rts   

SafeNumber
    ; The number is a valid section value
    move.l  #1, d2   
    rts
    
   
; Finds the section where the entered number starts and saves it in a0
FindSection
    ; Preserve section number by copying it to d2
    move.l  d1, d2

    ; Subtract 1 to achieve 0 indexing with our bin file 
    sub.l   #1, d2
    
    ; Multiply by the section size to find our offset
    mulu    #CONSTANT_SECTION_SIZE, d2
    
    ; Add the offset to the start of the data array
    move.l  (data_sections_start), a0
    add.l   d2, a0
    
    rts


; Finds metadata in the header for the chunk counts and saves it in a1
FindChunkSizes
    ; Preserve section number by copying it to d2
    move.l  d1, d2
    
    ; Subtract 1 to achieve 0 indexing with our file header 
    sub.l   #1, d2
    
    ; Multiply by the chunk per section size to find our offset in the header file
    lsl.l   #2, d2
    mulu.w  #CHUNKS_PER_SECTION, d2
        
    ; Skip past the id_field and the sections field in the bin file
    lea     binary_file, a1
    add.l   #SIZE_LONG, a1
    add.l   #SIZE_LONG, a1
    
    ; Add the offset to the beginning of the header file 
    add.l   d2, a1
    
    ; Save a1 in a3 as well, so we can iterate through chunk sizes more easily 
    move.l  a1, a3
    
    rts
    

**  MATH FUNCTIONS  **


; Sums up all the bytes in a chunk
ByteCountLoop
    ; Add to our going sum
    ; Loads byte separately before adding to the going sum to avoid overflow
    move.b  (a0)+, d7
    add.l   d7, d6
    
    ; Check if we've calculated everything
    sub.l   #1, d5
    bne     ByteCountLoop
    rts
    
    
; Sums up all the words in a chunk
WordCountLoop
    ; Add to our going sum
    ; Loads word separately before adding to the going sum to avoid overflow
    move.w  (a0)+, d7
    add.l   d7, d6
    
    ; Check if we've calculated everything
    sub.l   #1, d5
    bne     WordCountLoop
    rts


; Sums up all the longs in a chunk    
LongCountLoop
    ; Add to our going sum
    ; Loads long separately before adding to the going sum to avoid overflow
    move.l  (a0)+, d7
    add.l   d7, d6
    
    ; Check if we've calculated everything
    sub.l   #1, d5
    bne     LongCountLoop
    rts
    

; Sum up all of the elements in this chunk
SumChunk
    clr.l   d4  ; The total bytes for this section
    clr.l   d5  ; The counter for the total bytes
    clr.l   d6  ; The sum of the bytes, later the average of the bytes
    clr.l   d7  ; The temporary holder for the current byte
    
    ; Determine how many bytes we're working with from a3, our header pointer
    move.l  (a3), d4
    move.l  d4, d5
    
ByteChunk
    ; Sum bytes if we're in byte mode
    cmp.l   #SIZE_BYTE, d3
    bne     WordChunk
    jsr     ByteCountLoop
    rts
    
WordChunk
    ; Sum words if we're in word mode
    cmp.l   #SIZE_WORD, d3
    bne     LongChunk
    jsr     WordCountLoop
    rts
    
LongChunk
    ; Sum longs if we're in long mode
    cmp.l   #SIZE_LONG, d3
    bne     EscapeChunk
    jsr     LongCountLoop
    rts
    
EscapeChunk
    rts
    

; Calculates the average mean, with sum in d6 and number of items in d4
FindAverage
    ; Divide to get magnitude in and remainder
    divu    d4, d6
    
    ; Magnitude is in d7, remainder in d6
    move.w  d6, d7
    swap    d6
    ext.l   d6
    ext.l   d7
    
    ; d6 / d4 = x / 100, where x is the decimal value in hundredths
    ; We solve for x with cross-mulitplication
    
    ; Get it into the next point below our decimal accuracy for rounding purposes
    mulu    #ROUNDING_POINT_PLUS_1, d6
    divu    d4, d6 
    and.l   #BOTTOM_WORD_MASK, d6
    
    ; Add 5 and divide by 10 to round to the hundredths
    add.l   #5, d6
    divu    #10, d6
    and.l   #BOTTOM_WORD_MASK, d6
    
    ; If we went over our rounding value, then we add that value to our magnitude
    cmp.l   #ROUNDING_POINT, d6
    blt     AverageCalculated
    divu    #100, d6
    add.w   d6, d7
    swap    d6
    ext.l   d6
    
AverageCalculated
    rts
 
 
**  PRINT FUNCTIONS  **   


; Display the section number inputted by the user
DisplaySectionNumber

    ; Load section message
    lea     (section_message), a1
    move.l  #TASK_PRINT_MESSAGE_WITHOUT_CRLF,d0
    TRAP #15    

    ; Inputted number is already in d1
    move.l  #TASK_PRINT_NUMBER,d0
    TRAP #15   

    ; Newline
    lea     (empty), a1
    move.l  #TASK_PRINT_MESSAGE_WITH_CRLF,d0
    TRAP #15      
    
    rts
    
    
; Displays the averages for this chunk    
DisplayAverage
    ; Load the number of things (bytes/words/longs) in this chunk and print
    move.l  (a3), d1
    move.l  #TASK_PRINT_NUMBER,d0
    TRAP #15    
    
ByteDisplay
    ; Display bytes if we're in byte mode
    cmp.l   #SIZE_BYTE, d3
    bne     WordDisplay
    lea     (byte_message), a1
    bra     EscapeDisplay

WordDisplay
    ; Display words if we're in word mode
    cmp.l   #SIZE_WORD, d3
    bne     LongDisplay
    lea     (short_message), a1
    bra     EscapeDisplay
    
LongDisplay
    ; Display longs if we're in long mode
    cmp.l   #SIZE_LONG, d3
    bne     EscapeDisplay
    lea     (long_message), a1
    bra     EscapeDisplay

EscapeDisplay
    move.l  #TASK_PRINT_MESSAGE_WITHOUT_CRLF,d0
    TRAP #15  

    ; Display the magnitude component
    clr.l   d1
    move.l  d7, d1
    move.l  #TASK_PRINT_NUMBER,d0
    TRAP #15   
    
    ; Print the decimal point
    lea     (decimal), a1
    move.l  #TASK_PRINT_MESSAGE_WITHOUT_CRLF,d0
    TRAP #15 

    ; If our number is less than 10, we need a leading zero
    clr.l   d1
    cmp.l   #10, d6
    bgt     PrintDecimal
    move.l  #TASK_PRINT_NUMBER,d0
    TRAP #15  

PrintDecimal    
    ; Display the fractional component
    move.l  d6, d1
    move.l  #TASK_PRINT_NUMBER,d0
    TRAP #15   
    
    ; Newline
    lea     (empty), a1
    move.l  #TASK_PRINT_MESSAGE_WITH_CRLF,d0
    TRAP #15 
    
    rts  
    
    
; Prints an error message about the section input
OOBMessage
    lea (oob_message), a1
    move.l  #TASK_PRINT_MESSAGE_WITH_CRLF,d0
    TRAP #15    
    
    rts


; Prints an error message about the file's format
ErrorMessage
    lea (error_message), a1
    move.l  #TASK_PRINT_MESSAGE_WITH_CRLF,d0
    TRAP #15    
    
    move.l  #$9,d0
    TRAP #15
    
    rts


**  LOOP FUNCTIONS  **    
    
    
; Does the heavy lifting for our loop
LoopOperation
    ; Find some meta data in the header of the bin file
    jsr FindSection
    jsr DisplaySectionNumber
    jsr FindChunkSizes
    
    ; Set d3 to byte mode and sum this chunk
    move.l  #SIZE_BYTE, d3    
    jsr SumChunk
    jsr FindAverage
    jsr DisplayAverage
    
    ; Adjust offset to the word section
    add.l   #CONSTANT_BYTE_CHUNK_SIZE, a0
    move.l  (a3), d3
    sub.l   d3, a0

    ; Update our chunk size
    add.l   #SIZE_LONG, a3
    
    ; Set d3 to the word mode and sum this chunk
    move.l  #SIZE_WORD, d3   
    jsr SumChunk
    jsr FindAverage
    jsr DisplayAverage
    
    ; Adjust offset to the long section
    add.l   #CONSTANT_WORD_CHUNK_SIZE, a0
    move.l  (a3), d3
    lsl.l   #1, d3
    sub.l   d3, a0

    ; Update our chunk size
    add.l   #SIZE_LONG, a3
    
    ; Set d3 to the long mode and sum this chunk
    move.l  #SIZE_LONG, d3   
    jsr SumChunk
    jsr FindAverage
    jsr DisplayAverage
    
    rts


**  MAIN FUNCTION  **    
    
    
START:       
    ; Load the input file            
    jsr LoadBinFile

InputLoop
    lea (prompt_message), a1
    move.l  #TASK_PRINT_MESSAGE_WITHOUT_CRLF,d0
    TRAP #15    
    

    ; Get input from user
    move.l  #TASK_READ_INPUT, d0
    TRAP #15    
    
    ; Make sure the section number is safe
    jsr ValidateSection
    cmp.l   #1, d2
    bne     InputLoop
    
    ; Go to our loop once we have inputs!
    jsr LoopOperation
        
    bra InputLoop
    
    move.l  #$9,d0
    TRAP #15

* Variables and constants 

; Our id field guarantees that the file was created in good faith
id_field                dc.b    'DCRW'
; How many sections we're working with 
number_of_sections      ds.l    1
; The address at which the data sections start
data_sections_start     ds.l    1

; Messages for average value printing
byte_message        dc.b    ' bytes, average value ',0
short_message       dc.b    ' shorts, average value ',0
long_message        dc.b    ' longs, average value ',0

; Other messages for user communication and information display
section_message     dc.b    'Section ',0
decimal             dc.b    '.',0
empty               dc.b    '',0
prompt_message      dc.b    'Please enter which section you want: ',0
error_message       dc.b    'An error occured! Make sure that all of the input data is correctly formatted!.',0
oob_message         dc.b    'That is not a valid number!',0
                            

; Link to the binary file 
binary_file     INCBIN      "binary.bin"

    END    START        ; last line of source



    
*~Font name~Courier New~
*~Font size~12~
*~Tab type~1~
*~Tab size~4~
