#include "stdafx.h"
#include <string>
#include "Random.h"
#include "IO.h"

// Max and min possible values for the section amounts
// Need to be declared as constants for the purposes of creating arrays that the 68k file relies on 
const int sectionMaxPossible = 50;
const int sectionMinPossible = 0;
const int chunkSizeMaxPossible = 500;

// Files associated with this project
char initFileName[] = "DataChunkWriterIni.txt";
char binaryFileName[] = "binary.bin";
char statsFileName[] = "stats.txt";

// These arrays hold a bunch of pre-generated chunk sizes
int byteSectionAmounts[sectionMaxPossible];
int shortSectionAmounts[sectionMaxPossible];
int longSectionAmounts[sectionMaxPossible];

// Our outputs 
FILE * outputStat;
FILE * outputBin;

// Length of a char
char charLength = 8;
// Length of a short
char shortLength = 16;

// Variables for reading in data from the ini file 
char offsetLength = 8;
char lineLength = 20;

// ini file values
int sectionMin;
int sectionMax;
int byteNumberMin;
int byteNumberMax;
int shortNumberMin;
int shortNumberMax;
int longNumberMin;
int longNumberMax;
int byteValueMin;
int byteValueMax;
int shortValueMin;
int shortValueMax;
int longValueMin;
int longValueMax;

// Print 16 bit number, reversing its endianness
void printEndianConvert16Bit(FILE * file, int input)
{
	fprintf(file, "%c", input >> charLength);
	fprintf(file, "%c", input);
}

// Print 32 bit number, reversing its endianness
void printEndianConvert32Bit(FILE * file, int input)
{
	fprintf(file, "%c", input >> (shortLength + charLength));
	fprintf(file, "%c", input >> shortLength);
	fprintf(file, "%c", input >> charLength);
	fprintf(file, "%c", input);
}


// Read in the ini file and its data
int readIniFile()
{
	FILE* initFile = fopen(initFileName, "r");

	// Quit out if there's a file error
	if (!initFile)
		return -1;

	char line[1000];

	// Get the section min and max
	fgets(line, lineLength, initFile);
	sectionMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	sectionMax = atoi(line + offsetLength);

	// Get the number mins and maxes
	fgets(line, lineLength, initFile);
	byteNumberMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	byteNumberMax = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	shortNumberMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	shortNumberMax = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	longNumberMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	longNumberMax = atoi(line + offsetLength);

	// Get the value mins and maxes
	fgets(line, lineLength, initFile);
	byteValueMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	byteValueMax = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	shortValueMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	shortValueMax = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	longValueMin = atoi(line + offsetLength);
	fgets(line, lineLength, initFile);
	longValueMax = atoi(line + offsetLength);

	// validate the ini file's input 
	if (sectionMax > sectionMaxPossible ||
		sectionMin < sectionMinPossible ||
		sectionMax < sectionMin ||
		byteNumberMax < byteNumberMin ||
		shortNumberMax < shortNumberMin ||
		longNumberMax < longNumberMin ||
		byteValueMax - byteValueMin < 0 ||
		shortValueMax - shortValueMin < 0 ||
		longValueMax - longValueMin < 0 ||
		byteNumberMax > chunkSizeMaxPossible ||
		shortNumberMax > chunkSizeMaxPossible ||
		longNumberMax > chunkSizeMaxPossible ) {

		return -1;
	}

	// Return that everything is okay
	return 0;
}

// Get the number of sections from the user
int getUserInput() {

	// How many data sections to write 
	int sectionsOfData;

	// Prompt the user for data
	printf("Hey! How many sections of data do you want? It must be %d < n < %d.\n", sectionMin, sectionMax);
	printf("(Please note: if you enter multiple numbers, I'll only read the first one!)\n\n> ");

	// Prepare for bad input with a loop
	while (1) {

		// Read user input
		int itemsRead = scanf("%d", &sectionsOfData);

		//Bear through non-integer input
		while (getchar() != '\n');
		if (!itemsRead) {
			printf("Hey, that's not a number! What're you trying to pull here, man?\n");
			printf("> ");
			continue;
		}

		// Check for too small, be snide if it resolves to negative 
		if (sectionsOfData <= sectionMin) {
			printf("Hey, bozo! That's too few sections!\n");
			printf("Or way too many, if you overflowed the integer...\n");
			printf("Try again!\n");
			printf("> ");
			sectionsOfData = 0;
			continue;
		}
		// Check for too large
		else if (sectionsOfData >= sectionMax) {
			printf("Hey, bozo! That's too many sections! Try again!\n");
			printf("> ");
			sectionsOfData = 0;
			continue;
		}
		// Confirm the user's input
		printf("Okay, you chose this many sections: %d!\n", sectionsOfData);
		break;

	}

	return sectionsOfData;
}


// Generate our chunk sizes for potential sections
// Note: not every item in the SectionAmounts arrays will be used
void generateChunkSizes(int sections)
{
	// Open files 
	outputStat = fopen(statsFileName, "w");
	outputBin = fopen(binaryFileName, "wb");

	// Print section amount
	fprintf(outputStat, "User chose %d sections.\n", sections);

	// Print file header and sections 
	fprintf(outputBin, "DCRW");
	printEndianConvert32Bit(outputBin, sections);

	// Populate our chunk size lists for each possible section
	for (int i = 0; i < sectionMaxPossible; i++)
	{
		byteSectionAmounts[i] = getRandomLong(byteNumberMin, byteNumberMax);
		shortSectionAmounts[i] = getRandomLong(shortNumberMin, shortNumberMax);
		longSectionAmounts[i] = getRandomLong(longNumberMin, longNumberMax);
		if (i < sections)
		{
			printEndianConvert32Bit(outputBin, byteSectionAmounts[i]);
			printEndianConvert32Bit(outputBin, shortSectionAmounts[i]);
			printEndianConvert32Bit(outputBin, longSectionAmounts[i]);
		}

	}
}

// Calculates each data section and prints relevant information to the out files
int calculateAndPrintSections(int sections) 
{
	for (int i = 0; i < sections; i++)
	{
		// For each section, print the section number as a kind of header 
		fprintf(outputStat, "Section %d\n", i + 1);

		// Generate number of bytes 
		float averageValueByte = 0;
		int numOfBytes = byteSectionAmounts[i];

		// Generate that many bytes
		for (int j = 0; j < chunkSizeMaxPossible; j++)
		{
			// Write out as many random bytes as we need, then pad with 0s
			unsigned char cur = 0;
			if (j < numOfBytes) {
				cur = getRandomByte(byteValueMin, byteValueMax);
				averageValueByte += cur;
			}
			fprintf(outputBin, "%c", cur);
		}

		// Calculate and record average
		averageValueByte = roundToHundredths(averageValueByte / numOfBytes);
		fprintf(outputStat, "%d bytes, average value %0.2f\n", numOfBytes, averageValueByte);

		// Generate number of shorts 
		float averageValueShort = 0;
		int numOfShorts = shortSectionAmounts[i];

		// Generate that many shorts
		for (int j = 0; j < chunkSizeMaxPossible; j++)
		{
			// Write out as many random shorts as we need, then pad with 0s
			unsigned short cur = 0;
			if (j < numOfShorts) {
				cur = getRandomShort(shortValueMin, shortValueMax);
				averageValueShort += cur;
			}
			printEndianConvert16Bit(outputBin, cur);
		}

		// Calculate and record average
		averageValueShort = roundToHundredths(averageValueShort / numOfShorts);
		fprintf(outputStat, "%d shorts, average value %0.2f\n", numOfShorts, averageValueShort);

		// Generate number of longs
		float averageValueLong = 0;
		int numOfLongs = longSectionAmounts[i];

		// Generate that many longs
		for (int j = 0; j < chunkSizeMaxPossible; j++)
		{
			// Write out as many random longs as we need, then pad with 0s
			unsigned long cur = 0;
			if (j < numOfLongs) {
				cur = getRandomLong(longValueMin, longValueMax);
				averageValueLong += cur;
			}
			printEndianConvert32Bit(outputBin, cur);
		}

		// Calculate and record average
		averageValueLong = roundToHundredths(averageValueLong / numOfLongs);
		fprintf(outputStat, "%d longs, average value %0.2f\n", numOfLongs, averageValueLong);

	}

	// Return out that all went well
	return 0;

}

// Round inputted float to the nearest hundredth 
float roundToHundredths(float input)
{
	// Multiply by 100, round from the decimal, and then move it back 
	return round(input * 100.00f) / 100.00f;
}